using UnityEngine;

/*
 * Handler for Delegation Lines between Voter Agents in Visualised LD Simulations
 */
public class DelegationHandler : MonoBehaviour
{
    LineRenderer delegationLine;
    private VoterHandler voter;
    private VoterHandler guru;

    int numCurvePoints = 10;
    bool delegated = false;

    // Start is called before the first frame update, default line values set
    void Start()
    {
        delegationLine = new GameObject("Line").AddComponent<LineRenderer>();
        delegationLine.startColor = Color.black;
        delegationLine.endColor = Color.black;
        delegationLine.startWidth = 0.1f;
        delegationLine.endWidth = 0.1f;
        delegationLine.positionCount = numCurvePoints;
        delegationLine.useWorldSpace = true;
        delegationLine.enabled = false;
    }

    // Delegation line enabled as voter delegates, set between given delegator and delegate
    public void SetDelegation(VoterHandler voter, VoterHandler guru)
    {
        delegationLine.enabled = true;
        this.voter = voter;
        this.guru = guru;
        delegated = true;
    }

    // Delegation line disabled as voter removes delegation
    public void StopDelegation()
    {
        delegationLine.enabled = false;
        delegated = false;
    }

    // Line between delegator and delegate is made from 3 points to give it a curve
    void DrawDelegationCurve(Vector3 point0, Vector3 point1, Vector3 point2)
    {
        float t;
        Vector3 position;
        for (int i = 0; i < numCurvePoints; i++)
        {
            t = i / (numCurvePoints - 1.0f);
            position = (1.0f - t) * (1.0f - t) * point0
            + 2.0f * (1.0f - t) * t * point1 + t * t * point2;
            delegationLine.SetPosition(i, position);
        }
    }

    // Colour of delegation line set to colour of guru delegation leads to either directly or transitively
    public void setColour(Color guruColour)
    {
        Material delegationMat = new Material(Shader.Find("Standard"));
        delegationMat.color = guruColour;
        delegationLine.GetComponent<Renderer>().material = delegationMat;
    }

    // Update is called once per frame, if enabled line is constanly updated to stay between delegator and delegate for when they move
    void Update()
    {
        if (delegated)
        {
            float xMid;
            float zMid;
            if (voter.transform.position.x <= guru.transform.position.x)
            {
                xMid = voter.transform.position.x + (Mathf.Abs(voter.transform.position.x - guru.transform.position.x) / 2);
            }
            else
            {
                xMid = guru.transform.position.x + (Mathf.Abs(voter.transform.position.x - guru.transform.position.x) / 2);
            }

            if (voter.transform.position.z <= guru.transform.position.z)
            {
                zMid = voter.transform.position.z + (Mathf.Abs(voter.transform.position.z - guru.transform.position.z) / 2);
            }
            else
            {
                zMid = guru.transform.position.z + (Mathf.Abs(voter.transform.position.z - guru.transform.position.z) / 2);
            }

            Vector3 delegatorPos = voter.transform.position;
            Vector3 midPos = new Vector3(xMid, voter.transform.position.y + 1.5f, zMid);
            Vector3 guruPos = guru.transform.position;

            DrawDelegationCurve(delegatorPos, midPos, guruPos);
        }
    }

    // Delegation line removed from visualisation
    public void Remove()
    {
        delegated = false;
        Destroy(delegationLine);
        Destroy(gameObject);
        Destroy(this);
    }
}